/**
 * @license
 * Copyright 2017 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
/**
 * Returns the dimensions in the input shape that are broadcasted to
 * produce the provided output shape.
 *
 * The returned dimensions are 0-indexed and sorted. An example:
 * inShape = [4, 1, 3]
 * outShape = [5, 4, 3, 3]
 * result = [1]. Dimension 1 (2nd dimension of input) gets broadcasted 1 => 3.
 */
export function getBroadcastDims(inShape, outShape) {
    const inRank = inShape.length;
    const dims = [];
    for (let i = 0; i < inRank; i++) {
        const dim = inRank - 1 - i;
        const a = inShape[dim] || 1;
        const b = outShape[outShape.length - 1 - i] || 1;
        if (b > 1 && a === 1) {
            dims.unshift(dim);
        }
    }
    return dims;
}
/**
 * Returns the axes in the output space that should be reduced to produce
 * the input space.
 */
export function getReductionAxes(inShape, outShape) {
    const result = [];
    for (let i = 0; i < outShape.length; i++) {
        const inDim = inShape[inShape.length - i - 1];
        const outAxis = outShape.length - i - 1;
        const outDim = outShape[outAxis];
        if (inDim == null || (inDim === 1 && outDim > 1)) {
            result.unshift(outAxis);
        }
    }
    return result;
}
export function assertAndGetBroadcastShape(shapeA, shapeB) {
    const l = Math.max(shapeA.length, shapeB.length);
    const result = new Array(l);
    for (let i = 0; i < l; i++) {
        let a = shapeA[shapeA.length - i - 1];
        if (a == null) {
            a = 1;
        }
        let b = shapeB[shapeB.length - i - 1];
        if (b == null) {
            b = 1;
        }
        if (a === 1) {
            result[l - i - 1] = b;
        }
        else if (b === 1) {
            result[l - i - 1] = a;
        }
        else if (a !== b) {
            const errMsg = `Operands could not be broadcast together with shapes ` +
                `${shapeA} and ${shapeB}.`;
            throw Error(errMsg);
        }
        else {
            result[l - i - 1] = a;
        }
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnJvYWRjYXN0X3V0aWwuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi90ZmpzLWNvcmUvc3JjL29wcy9icm9hZGNhc3RfdXRpbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFFSDs7Ozs7Ozs7R0FRRztBQUNILE1BQU0sVUFBVSxnQkFBZ0IsQ0FDNUIsT0FBaUIsRUFBRSxRQUFrQjtJQUN2QyxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDO0lBQzlCLE1BQU0sSUFBSSxHQUFhLEVBQUUsQ0FBQztJQUMxQixLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFO1FBQy9CLE1BQU0sR0FBRyxHQUFHLE1BQU0sR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQzNCLE1BQU0sQ0FBQyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDNUIsTUFBTSxDQUFDLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUNwQixJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ25CO0tBQ0Y7SUFDRCxPQUFPLElBQUksQ0FBQztBQUNkLENBQUM7QUFFRDs7O0dBR0c7QUFDSCxNQUFNLFVBQVUsZ0JBQWdCLENBQzVCLE9BQWlCLEVBQUUsUUFBa0I7SUFDdkMsTUFBTSxNQUFNLEdBQWEsRUFBRSxDQUFDO0lBQzVCLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFO1FBQ3hDLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUM5QyxNQUFNLE9BQU8sR0FBRyxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDeEMsTUFBTSxNQUFNLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ2pDLElBQUksS0FBSyxJQUFJLElBQUksSUFBSSxDQUFDLEtBQUssS0FBSyxDQUFDLElBQUksTUFBTSxHQUFHLENBQUMsQ0FBQyxFQUFFO1lBQ2hELE1BQU0sQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDekI7S0FDRjtJQUNELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFFRCxNQUFNLFVBQVUsMEJBQTBCLENBQ3RDLE1BQWdCLEVBQUUsTUFBZ0I7SUFDcEMsTUFBTSxDQUFDLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUNqRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUU1QixLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFO1FBQzFCLElBQUksQ0FBQyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsSUFBSSxJQUFJLEVBQUU7WUFDYixDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ1A7UUFDRCxJQUFJLENBQUMsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLElBQUksSUFBSSxFQUFFO1lBQ2IsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNQO1FBQ0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ1gsTUFBTSxDQUFDLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZCO2FBQU0sSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ2xCLE1BQU0sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN2QjthQUFNLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUNsQixNQUFNLE1BQU0sR0FBRyx1REFBdUQ7Z0JBQ2xFLEdBQUcsTUFBTSxRQUFRLE1BQU0sR0FBRyxDQUFDO1lBQy9CLE1BQU0sS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3JCO2FBQU07WUFDTCxNQUFNLENBQUMsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDdkI7S0FDRjtJQUNELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgMjAxNyBHb29nbGUgTExDLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKTtcbiAqIHlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2Ugd2l0aCB0aGUgTGljZW5zZS5cbiAqIFlvdSBtYXkgb2J0YWluIGEgY29weSBvZiB0aGUgTGljZW5zZSBhdFxuICpcbiAqIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqIFVubGVzcyByZXF1aXJlZCBieSBhcHBsaWNhYmxlIGxhdyBvciBhZ3JlZWQgdG8gaW4gd3JpdGluZywgc29mdHdhcmVcbiAqIGRpc3RyaWJ1dGVkIHVuZGVyIHRoZSBMaWNlbnNlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuIFwiQVMgSVNcIiBCQVNJUyxcbiAqIFdJVEhPVVQgV0FSUkFOVElFUyBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBlaXRoZXIgZXhwcmVzcyBvciBpbXBsaWVkLlxuICogU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zIGFuZFxuICogbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKiA9PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PT09PVxuICovXG5cbi8qKlxuICogUmV0dXJucyB0aGUgZGltZW5zaW9ucyBpbiB0aGUgaW5wdXQgc2hhcGUgdGhhdCBhcmUgYnJvYWRjYXN0ZWQgdG9cbiAqIHByb2R1Y2UgdGhlIHByb3ZpZGVkIG91dHB1dCBzaGFwZS5cbiAqXG4gKiBUaGUgcmV0dXJuZWQgZGltZW5zaW9ucyBhcmUgMC1pbmRleGVkIGFuZCBzb3J0ZWQuIEFuIGV4YW1wbGU6XG4gKiBpblNoYXBlID0gWzQsIDEsIDNdXG4gKiBvdXRTaGFwZSA9IFs1LCA0LCAzLCAzXVxuICogcmVzdWx0ID0gWzFdLiBEaW1lbnNpb24gMSAoMm5kIGRpbWVuc2lvbiBvZiBpbnB1dCkgZ2V0cyBicm9hZGNhc3RlZCAxID0+IDMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBnZXRCcm9hZGNhc3REaW1zKFxuICAgIGluU2hhcGU6IG51bWJlcltdLCBvdXRTaGFwZTogbnVtYmVyW10pOiBudW1iZXJbXSB7XG4gIGNvbnN0IGluUmFuayA9IGluU2hhcGUubGVuZ3RoO1xuICBjb25zdCBkaW1zOiBudW1iZXJbXSA9IFtdO1xuICBmb3IgKGxldCBpID0gMDsgaSA8IGluUmFuazsgaSsrKSB7XG4gICAgY29uc3QgZGltID0gaW5SYW5rIC0gMSAtIGk7XG4gICAgY29uc3QgYSA9IGluU2hhcGVbZGltXSB8fCAxO1xuICAgIGNvbnN0IGIgPSBvdXRTaGFwZVtvdXRTaGFwZS5sZW5ndGggLSAxIC0gaV0gfHwgMTtcbiAgICBpZiAoYiA+IDEgJiYgYSA9PT0gMSkge1xuICAgICAgZGltcy51bnNoaWZ0KGRpbSk7XG4gICAgfVxuICB9XG4gIHJldHVybiBkaW1zO1xufVxuXG4vKipcbiAqIFJldHVybnMgdGhlIGF4ZXMgaW4gdGhlIG91dHB1dCBzcGFjZSB0aGF0IHNob3VsZCBiZSByZWR1Y2VkIHRvIHByb2R1Y2VcbiAqIHRoZSBpbnB1dCBzcGFjZS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldFJlZHVjdGlvbkF4ZXMoXG4gICAgaW5TaGFwZTogbnVtYmVyW10sIG91dFNoYXBlOiBudW1iZXJbXSk6IG51bWJlcltdIHtcbiAgY29uc3QgcmVzdWx0OiBudW1iZXJbXSA9IFtdO1xuICBmb3IgKGxldCBpID0gMDsgaSA8IG91dFNoYXBlLmxlbmd0aDsgaSsrKSB7XG4gICAgY29uc3QgaW5EaW0gPSBpblNoYXBlW2luU2hhcGUubGVuZ3RoIC0gaSAtIDFdO1xuICAgIGNvbnN0IG91dEF4aXMgPSBvdXRTaGFwZS5sZW5ndGggLSBpIC0gMTtcbiAgICBjb25zdCBvdXREaW0gPSBvdXRTaGFwZVtvdXRBeGlzXTtcbiAgICBpZiAoaW5EaW0gPT0gbnVsbCB8fCAoaW5EaW0gPT09IDEgJiYgb3V0RGltID4gMSkpIHtcbiAgICAgIHJlc3VsdC51bnNoaWZ0KG91dEF4aXMpO1xuICAgIH1cbiAgfVxuICByZXR1cm4gcmVzdWx0O1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYXNzZXJ0QW5kR2V0QnJvYWRjYXN0U2hhcGUoXG4gICAgc2hhcGVBOiBudW1iZXJbXSwgc2hhcGVCOiBudW1iZXJbXSk6IG51bWJlcltdIHtcbiAgY29uc3QgbCA9IE1hdGgubWF4KHNoYXBlQS5sZW5ndGgsIHNoYXBlQi5sZW5ndGgpO1xuICBjb25zdCByZXN1bHQgPSBuZXcgQXJyYXkobCk7XG5cbiAgZm9yIChsZXQgaSA9IDA7IGkgPCBsOyBpKyspIHtcbiAgICBsZXQgYSA9IHNoYXBlQVtzaGFwZUEubGVuZ3RoIC0gaSAtIDFdO1xuICAgIGlmIChhID09IG51bGwpIHtcbiAgICAgIGEgPSAxO1xuICAgIH1cbiAgICBsZXQgYiA9IHNoYXBlQltzaGFwZUIubGVuZ3RoIC0gaSAtIDFdO1xuICAgIGlmIChiID09IG51bGwpIHtcbiAgICAgIGIgPSAxO1xuICAgIH1cbiAgICBpZiAoYSA9PT0gMSkge1xuICAgICAgcmVzdWx0W2wgLSBpIC0gMV0gPSBiO1xuICAgIH0gZWxzZSBpZiAoYiA9PT0gMSkge1xuICAgICAgcmVzdWx0W2wgLSBpIC0gMV0gPSBhO1xuICAgIH0gZWxzZSBpZiAoYSAhPT0gYikge1xuICAgICAgY29uc3QgZXJyTXNnID0gYE9wZXJhbmRzIGNvdWxkIG5vdCBiZSBicm9hZGNhc3QgdG9nZXRoZXIgd2l0aCBzaGFwZXMgYCArXG4gICAgICAgICAgYCR7c2hhcGVBfSBhbmQgJHtzaGFwZUJ9LmA7XG4gICAgICB0aHJvdyBFcnJvcihlcnJNc2cpO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXN1bHRbbCAtIGkgLSAxXSA9IGE7XG4gICAgfVxuICB9XG4gIHJldHVybiByZXN1bHQ7XG59XG4iXX0=